// Database sederhana untuk menyimpan data SIAKAD
// Dalam implementasi nyata, ini akan diganti dengan database sesungguhnya

let users = [
  // Data admin default
  {
    id: 1,
    fullName: "Admin SMK Tanjung",
    email: "admin@smktanjung.ac.id",
    phone: "081234567890",
    role: "admin",
    password: "admin123",
    major: null,
    status: "aktif",
    createdAt: new Date().toISOString()
  },
  // Data guru default
  {
    id: 2,
    fullName: "Budi Santoso",
    email: "budi@smktanjung.ac.id",
    phone: "081234567892",
    role: "guru",
    password: "guru123",
    major: null,
    status: "aktif",
    createdAt: new Date().toISOString()
  },
  {
    id: 4,
    fullName: "Sari Indah",
    email: "sari@smktanjung.ac.id",
    phone: "081234567893",
    role: "guru",
    password: "guru123",
    major: null,
    status: "aktif",
    createdAt: new Date().toISOString()
  },
  // Data siswa default
  {
    id: 3,
    fullName: "Ahmad Rizki",
    email: "ahmad@smktanjung.ac.id",
    phone: "081234567890",
    role: "siswa",
    password: "siswa123",
    major: "rpl",
    status: "aktif",
    createdAt: new Date().toISOString()
  }
];

// Data siswa
let students = [
  {
    id: 1,
    nis: "2024001",
    fullName: "Ahmad Rizki",
    email: "ahmad@smktanjung.ac.id",
    phone: "081234567890",
    address: "Jl. Merdeka No. 1, Jakarta",
    class: "X RPL 1",
    major: "rpl",
    gender: "laki-laki",
    religion: "islam",
    status: "aktif",
    createdAt: new Date().toISOString()
  },
  {
    id: 2,
    nis: "2024002",
    fullName: "Siti Nurhaliza",
    email: "siti@smktanjung.ac.id",
    phone: "081234567891",
    address: "Jl. Sudirman No. 2, Jakarta",
    class: "X AKL 1",
    major: "akl",
    gender: "perempuan",
    religion: "islam",
    status: "aktif",
    createdAt: new Date().toISOString()
  }
];

// Data guru
let teachers = [
  {
    id: 1,
    nip: "196501011990031001",
    fullName: "Budi Santoso",
    email: "budi@smktanjung.ac.id",
    phone: "081234567892",
    subjects: ["Matematika", "Fisika"],
    class: "X RPL 1",
    status: "aktif",
    createdAt: new Date().toISOString()
  },
  {
    id: 2,
    nip: "196502021990032002",
    fullName: "Sari Indah",
    email: "sari@smktanjung.ac.id",
    phone: "081234567893",
    subjects: ["Bahasa Indonesia", "Bahasa Inggris"],
    class: "X AKL 1",
    status: "aktif",
    createdAt: new Date().toISOString()
  }
];

// Data kelas
let classes = [
  {
    id: 1,
    name: "X RPL 1",
    major: "rpl",
    homeroomTeacher: "Budi Santoso",
    studentCount: 30,
    status: "aktif"
  },
  {
    id: 2,
    name: "X AKL 1",
    major: "akl",
    homeroomTeacher: "Sari Indah",
    studentCount: 28,
    status: "aktif"
  }
];

// Data jadwal
let schedules = [
  {
    id: 1,
    class: "X RPL 1",
    day: "Senin",
    time: "07:00-08:30",
    subject: "Matematika",
    teacher: "Budi Santoso",
    room: "R. 101"
  },
  {
    id: 2,
    class: "X RPL 1",
    day: "Senin",
    time: "08:30-10:00",
    subject: "Bahasa Indonesia",
    teacher: "Sari Indah",
    room: "R. 101"
  }
];

// Data nilai
let grades = [
  {
    id: 1,
    studentId: 1,
    studentName: "Ahmad Rizki",
    subject: "Matematika",
    teacher: "Budi Santoso",
    class: "X RPL 1",
    dailyGrade: 85,        // Nilai Harian/Tugas
    midtermGrade: 80,      // UTS
    finalGrade: 88,        // UAS
    attitudeGrade: "Baik", // Nilai Sikap (Sangat Baik, Baik, Cukup)
    average: 84.33,
    semester: "1",
    year: "2024/2025"
  }
];

// Data kehadiran
let attendance = [
  {
    id: 1,
    studentId: 1,
    studentName: "Ahmad Rizki",
    class: "X RPL 1",
    date: "2024-09-14",
    status: "hadir", // hadir, sakit, izin, alpa
    subject: "Matematika",
    teacher: "Budi Santoso"
  }
];

// Penugasan siswa ke mata pelajaran tertentu (opsional, selain dari jadwal)
// Bentuk: { id, studentId, subject, class, teacher }
let subjectAssignments = [];

// Data tagihan/pembayaran siswa
let studentPayments = [
  {
    id: 1,
    studentId: 1,
    period: "2024/2025 - Ganjil",
    billType: "SPP Bulanan",
    startDate: "2024-07-01",
    endDate: "2024-07-10",
    status: "BB", // BB = Belum Bayar, L = Lunas
    amount: 300000,
    discount: 0,
    fine: 0,
    total: 300000,
    paidDate: null
  }
];

// Jurusan yang tersedia di SMK Tanjung
export const majors = [
  { id: "mplb", name: "MPLB", fullName: "Manajemen Perkantoran dan Layanan Bisnis" },
  { id: "akl", name: "AKL", fullName: "Akuntansi dan Keuangan Lembaga" },
  { id: "dkv", name: "DKV", fullName: "Desain Komunikasi Visual" },
  { id: "rpl", name: "RPL", fullName: "Rekayasa Perangkat Lunak" },
  { id: "tkkr", name: "TKKR", fullName: "Teknik Kecantikan Kulit dan Rambut" }
];

// Jenis kelamin
export const genderOptions = [
  { id: "laki-laki", name: "Laki-laki" },
  { id: "perempuan", name: "Perempuan" }
];

// Agama resmi di Indonesia
export const religionOptions = [
  { id: "islam", name: "Islam" },
  { id: "kristen", name: "Kristen" },
  { id: "katolik", name: "Katolik" },
  { id: "hindu", name: "Hindu" },
  { id: "buddha", name: "Buddha" }
];

// Daftar mata pelajaran dasar dan spesifik jurusan
const coreSubjects = [
  "Matematika",
  "Bahasa Indonesia",
  "Bahasa Inggris",
  "Pendidikan Pancasila",
  "Pendidikan Agama",
  "PJOK",
  "Sejarah",
  "Seni Budaya",
  "Informatika"
];

const vocationalSubjects = {
  mplb: [
    "Administrasi Umum",
    "Aplikasi Perkantoran",
    "Korespondensi",
    "Otomatisasi Tata Kelola Kepegawaian"
  ],
  akl: [
    "Akuntansi Perusahaan Dagang",
    "Akuntansi Perusahaan Jasa",
    "Perbankan Dasar",
    "Administrasi Pajak"
  ],
  dkv: [
    "Dasar Desain Komunikasi Visual",
    "Fotografi",
    "Animasi 2D",
    "   "
  ],
  rpl: [
    "Dasar Pemrograman",
    "Pemrograman Web",
    "Basis Data",
    "Pemodelan Perangkat Lunak"
  ],
  tkkr: [
    "Dasar Kecantikan Kulit",
    "Dasar Kecantikan Rambut",
    "Perawatan Wajah",
    "Tata Rias Pengantin"
  ]
};

export function getSubjectsByMajor(majorId) {
  if (!majorId || majorId === "all") {
    return getAllAvailableSubjects();
  }
  const specialized = vocationalSubjects[majorId] || [];
  return Array.from(new Set([...coreSubjects, ...specialized]));
}

export function getAllAvailableSubjects() {
  const subjectSet = new Set(coreSubjects);
  Object.values(vocationalSubjects).forEach(list => {
    list.forEach(subject => subjectSet.add(subject));
  });
  return Array.from(subjectSet);
}

// ===== Helper integrasi antar entitas =====
export function getStudentByEmail(email) {
  return students.find(s => s.email === email);
}

export function getTeacherByEmail(email) {
  return teachers.find(t => t.email === email);
}

export function updateUserByEmail(email, updateData) {
  const idx = users.findIndex(u => u.email === email);
  if (idx !== -1) {
    users[idx] = { ...users[idx], ...updateData };
    saveState(); // Simpan perubahan ke localStorage
    return users[idx];
  }
  return null;
}

function ensureLinkedUserForStudent(student) {
  console.log('ensureLinkedUserForStudent called with:', student);
  const existing = users.find(u => u.email === student.email);
  if (!existing) {
    const newUser = {
      id: users.length + 1,
      fullName: student.fullName,
      email: student.email,
      phone: student.phone || '',
      role: 'siswa',
      password: student.nis || 'siswa123',
      major: student.major || null,
      status: student.status || 'aktif',
      createdAt: new Date().toISOString()
    };
    users.push(newUser);
    console.log('Created new user for student:', newUser);
    saveState(); // Simpan perubahan ke localStorage
  } else {
    // sinkron nama/telepon/major agar konsisten
    existing.fullName = student.fullName;
    existing.phone = student.phone || existing.phone;
    existing.major = student.major || existing.major;
    console.log('Updated existing user for student:', existing);
    saveState(); // Simpan perubahan ke localStorage
  }
}

function ensureLinkedUserForTeacher(teacher) {
  const existing = users.find(u => u.email === teacher.email);
  if (!existing) {
    users.push({
      id: users.length + 1,
      fullName: teacher.fullName,
      email: teacher.email,
      phone: teacher.phone || '',
      role: 'guru',
      password: teacher.nip || 'guru123', // Gunakan NIP sebagai password default
      major: null,
      status: teacher.status || 'aktif',
      createdAt: new Date().toISOString()
    });
    saveState(); // Simpan perubahan ke localStorage
  } else {
    // Update data yang ada
    existing.fullName = teacher.fullName;
    existing.phone = teacher.phone || existing.phone;
    existing.status = teacher.status || existing.status;
    // Update password jika NIP berubah
    if (teacher.nip && existing.password !== teacher.nip) {
      existing.password = teacher.nip;
    }
    saveState();
  }
}

// Fungsi untuk menambah user baru
function saveState() {
  try {
    if (typeof localStorage === 'undefined') return;
    const state = { users, students, teachers, classes, schedules, grades, attendance, subjectAssignments, studentPayments };
    localStorage.setItem('siakad_state', JSON.stringify(state));
    console.log('Data saved to localStorage at:', new Date().toLocaleTimeString());
  } catch (error) { 
    console.error('Error saving to localStorage:', error);
  }
}

function loadState() {
  try {
    if (typeof localStorage === 'undefined') return;
    const raw = localStorage.getItem('siakad_state');
    if (!raw) return;
    const state = JSON.parse(raw);
    if (state.users) users = state.users;
    if (state.students) students = state.students;
    if (state.teachers) teachers = state.teachers;
    if (state.classes) classes = state.classes;
    if (state.schedules) schedules = state.schedules;
    if (state.grades) grades = state.grades;
    if (state.attendance) attendance = state.attendance;
    if (state.subjectAssignments) subjectAssignments = state.subjectAssignments;
    if (state.studentPayments) studentPayments = state.studentPayments;
  } catch (_) { /* noop */ }
}

// Load persisted state once on module init (client-side only)
try { 
  loadState(); 
  // Pastikan semua guru dan siswa memiliki akun login
  teachers.forEach(ensureLinkedUserForTeacher);
  students.forEach(ensureLinkedUserForStudent);
} catch (_) {}

export function forceReloadFromStorage() { loadState(); }

export function addUser(userData) {
  const newUser = {
    id: users.length + 1,
    ...userData,
    status: "aktif",
    createdAt: new Date().toISOString()
  };
  
  users.push(newUser);
  saveState();
  return newUser;
}

// Fungsi untuk login (cek email dan password)
export function loginUser(email, password) {
  const trimmedEmail = (email || '').trim();
  const trimmedPass = (password || '').trim();
  
  // Exact match first - cek di data users
  let user = users.find(u => u.email === trimmedEmail && (u.password === trimmedPass));
  if (user) return user;

  // Fallback: cek apakah password adalah NIP untuk guru
  const teacher = teachers.find(t => t.email === trimmedEmail);
  if (teacher && String(teacher.nip) === trimmedPass) {
    // Pastikan ada user account untuk guru ini
    let linkedUser = users.find(u => u.email === trimmedEmail);
    if (!linkedUser) {
      // Buat user account jika belum ada
      ensureLinkedUserForTeacher(teacher);
      linkedUser = users.find(u => u.email === trimmedEmail);
    }
    return linkedUser;
  }
  
  // Fallback: cek apakah password adalah NIS untuk siswa
  const student = students.find(s => s.email === trimmedEmail);
  if (student && String(student.nis) === trimmedPass) {
    let linkedUser = users.find(u => u.email === trimmedEmail);
    if (!linkedUser) {
      ensureLinkedUserForStudent(student);
      linkedUser = users.find(u => u.email === trimmedEmail);
    }
    return linkedUser;
  }
  
  return null;
}

// Fungsi untuk mendapatkan semua user (untuk admin)
export function getAllUsers() {
  return users;
}

// Fungsi untuk mendapatkan user by ID
export function getUserById(id) {
  return users.find(u => u.id === id);
}

// Fungsi untuk update user
export function updateUser(id, updateData) {
  console.log('updateUser called with id:', id, 'updateData:', updateData);
  const userIndex = users.findIndex(u => u.id === id);
  console.log('User index found:', userIndex);
  if (userIndex !== -1) {
    const oldUser = users[userIndex];
    users[userIndex] = { ...users[userIndex], ...updateData };
    console.log('User updated from:', oldUser, 'to:', users[userIndex]);
    saveState(); // Simpan perubahan ke localStorage
    return users[userIndex];
  }
  console.log('User not found with id:', id);
  return null;
}

// Fungsi untuk delete user
export function deleteUser(id) {
  const userIndex = users.findIndex(u => u.id === id);
  if (userIndex !== -1) {
    return users.splice(userIndex, 1)[0];
  }
  return null;
}

// Fungsi untuk cek apakah email sudah terdaftar
export function isEmailExists(email) {
  return users.some(u => u.email === email);
}

// ===== FUNGSI UNTUK DATA SISWA =====
export function getAllStudents() {
  return students;
}

export function getStudentById(id) {
  return students.find(s => s.id === id);
}

export function addStudent(studentData) {
  const newStudent = {
    id: students.length + 1,
    ...studentData,
    status: "aktif",
    createdAt: new Date().toISOString()
  };
  students.push(newStudent);
  saveState();
  ensureLinkedUserForStudent(newStudent);
  return newStudent;
}

export function updateStudent(id, updateData) {
  const studentIndex = students.findIndex(s => s.id === id);
  if (studentIndex !== -1) {
    students[studentIndex] = { ...students[studentIndex], ...updateData };
    ensureLinkedUserForStudent(students[studentIndex]);
    saveState();
    return students[studentIndex];
  }
  return null;
}

export function deleteStudent(id) {
  const studentIndex = students.findIndex(s => s.id === id);
  if (studentIndex !== -1) {
    const removed = students.splice(studentIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK DATA GURU =====
export function getAllTeachers() {
  return teachers;
}

export function getTeacherById(id) {
  return teachers.find(t => t.id === id);
}

export function addTeacher(teacherData) {
  const newTeacher = {
    id: teachers.length + 1,
    ...teacherData,
    status: "aktif",
    createdAt: new Date().toISOString()
  };
  teachers.push(newTeacher);
  saveState();
  ensureLinkedUserForTeacher(newTeacher);
  return newTeacher;
}

export function updateTeacher(id, updateData) {
  const teacherIndex = teachers.findIndex(t => t.id === id);
  if (teacherIndex !== -1) {
    teachers[teacherIndex] = { ...teachers[teacherIndex], ...updateData };
    ensureLinkedUserForTeacher(teachers[teacherIndex]);
    saveState();
    return teachers[teacherIndex];
  }
  return null;
}

export function deleteTeacher(id) {
  const teacherIndex = teachers.findIndex(t => t.id === id);
  if (teacherIndex !== -1) {
    const removed = teachers.splice(teacherIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK DATA KELAS =====
export function getAllClasses() {
  return classes;
}

export function getClassById(id) {
  return classes.find(c => c.id === id);
}

export function addClass(classData) {
  const newClass = {
    id: classes.length + 1,
    ...classData,
    status: "aktif"
  };
  classes.push(newClass);
  saveState();
  return newClass;
}

export function updateClass(id, updateData) {
  const classIndex = classes.findIndex(c => c.id === id);
  if (classIndex !== -1) {
    classes[classIndex] = { ...classes[classIndex], ...updateData };
    saveState();
    return classes[classIndex];
  }
  return null;
}

export function deleteClass(id) {
  const classIndex = classes.findIndex(c => c.id === id);
  if (classIndex !== -1) {
    const removed = classes.splice(classIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK DATA JADWAL =====
export function getAllSchedules() {
  return schedules;
}

export function getScheduleByClass(className) {
  return schedules.filter(s => s.class === className);
}

export function addSchedule(scheduleData) {
  const newSchedule = {
    id: schedules.length + 1,
    ...scheduleData
  };
  schedules.push(newSchedule);
  saveState();
  return newSchedule;
}

export function updateSchedule(id, updateData) {
  const scheduleIndex = schedules.findIndex(s => s.id === id);
  if (scheduleIndex !== -1) {
    schedules[scheduleIndex] = { ...schedules[scheduleIndex], ...updateData };
    saveState();
    return schedules[scheduleIndex];
  }
  return null;
}

export function deleteSchedule(id) {
  const scheduleIndex = schedules.findIndex(s => s.id === id);
  if (scheduleIndex !== -1) {
    const removed = schedules.splice(scheduleIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK DATA NILAI =====
export function getAllGrades() {
  return grades;
}

export function getGradesByStudent(studentId) {
  return grades.filter(g => g.studentId === studentId);
}

export function getGradesByClass(className) {
  return grades.filter(g => g.class === className);
}

export function addGrade(gradeData) {
  const newGrade = {
    id: grades.length + 1,
    ...gradeData,
    average: calculateAverage(gradeData.dailyGrade, gradeData.midtermGrade, gradeData.finalGrade, gradeData.attitudeGrade)
  };
  grades.push(newGrade);
  saveState();
  return newGrade;
}

export function updateGrade(id, updateData) {
  const gradeIndex = grades.findIndex(g => g.id === id);
  if (gradeIndex !== -1) {
    const updatedGrade = { ...grades[gradeIndex], ...updateData };
    updatedGrade.average = calculateAverage(updatedGrade.dailyGrade, updatedGrade.midtermGrade, updatedGrade.finalGrade, updatedGrade.attitudeGrade);
    grades[gradeIndex] = updatedGrade;
    saveState();
    return grades[gradeIndex];
  }
  return null;
}

export function deleteGrade(id) {
  const gradeIndex = grades.findIndex(g => g.id === id);
  if (gradeIndex !== -1) {
    const removed = grades.splice(gradeIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

function calculateAverage(daily, midterm, final, attitude) {
  // Konversi nilai sikap ke angka untuk perhitungan
  const attitudeScore = attitude === "Sangat Baik" ? 95 : 
                       attitude === "Baik" ? 85 : 
                       attitude === "Cukup" ? 70 : 60;
  
  return ((daily * 0.3) + (midterm * 0.3) + (final * 0.3) + (attitudeScore * 0.1)).toFixed(2);
}

// ===== FUNGSI UNTUK DATA KEHADIRAN =====
export function getAllAttendance() {
  return attendance;
}

export function getAttendanceByStudent(studentId) {
  return attendance.filter(a => a.studentId === studentId);
}

export function getAttendanceByClass(className) {
  return attendance.filter(a => a.class === className);
}

export function addAttendance(attendanceData) {
  const newAttendance = {
    id: attendance.length + 1,
    ...attendanceData
  };
  attendance.push(newAttendance);
  saveState();
  return newAttendance;
}

export function updateAttendance(id, updateData) {
  const attendanceIndex = attendance.findIndex(a => a.id === id);
  if (attendanceIndex !== -1) {
    attendance[attendanceIndex] = { ...attendance[attendanceIndex], ...updateData };
    saveState();
    return attendance[attendanceIndex];
  }
  return null;
}

export function deleteAttendance(id) {
  const attendanceIndex = attendance.findIndex(a => a.id === id);
  if (attendanceIndex !== -1) {
    const removed = attendance.splice(attendanceIndex, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK TAGIHAN/PAYMENT SISWA =====
export function getAllStudentPayments() {
  return studentPayments;
}

export function getPaymentsByStudent(studentId) {
  const idNum = parseInt(studentId);
  return studentPayments.filter(p => p.studentId === idNum);
}

export function addStudentPayment(paymentData) {
  const baseAmount = Number(paymentData.amount) || 0;
  const discount = Number(paymentData.discount) || 0;
  const fine = Number(paymentData.fine) || 0;
  const total = Math.max(0, baseAmount - discount + fine);
  
  const newPayment = {
    id: studentPayments.length + 1,
    studentId: typeof paymentData.studentId === 'string' ? parseInt(paymentData.studentId) : paymentData.studentId,
    period: paymentData.period,
    billType: paymentData.billType,
    startDate: paymentData.startDate,
    endDate: paymentData.endDate || null,
    status: paymentData.status || 'BB',
    amount: baseAmount,
    discount,
    fine,
    total,
    paidDate: (paymentData.status === 'L')
      ? (paymentData.paidDate || new Date().toISOString().split('T')[0])
      : (paymentData.paidDate || null)
  };
  studentPayments.push(newPayment);
  saveState();
  return newPayment;
}

export function updateStudentPayment(id, updateData) {
  const index = studentPayments.findIndex(p => p.id === id);
  if (index === -1) return null;
  
  const merged = {
    ...studentPayments[index],
    ...updateData
  };
  
  const amount = Number(merged.amount) || 0;
  const discount = Number(merged.discount) || 0;
  const fine = Number(merged.fine) || 0;
  merged.total = Math.max(0, amount - discount + fine);
  
  if (merged.status === 'L') {
    merged.paidDate = merged.paidDate || new Date().toISOString().split('T')[0];
  } else {
    merged.paidDate = null;
  }
  
  studentPayments[index] = merged;
  saveState();
  return merged;
}

export function deleteStudentPayment(id) {
  const index = studentPayments.findIndex(p => p.id === id);
  if (index !== -1) {
    const removed = studentPayments.splice(index, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

// ===== FUNGSI UNTUK LAPORAN =====
export function getStudentRanking(className) {
  const classGrades = getGradesByClass(className);
  return classGrades.sort((a, b) => b.average - a.average);
}

export function getAttendanceReport(className, startDate, endDate) {
  const classAttendance = getAttendanceByClass(className);
  return classAttendance.filter(a => {
    const date = new Date(a.date);
    return date >= new Date(startDate) && date <= new Date(endDate);
  });
}

// ===== Helper tambahan untuk integrasi panel guru =====
export function getStudentsByClass(className) {
  return students.filter(s => s.class === className);
}

export function getSchedulesByTeacher(teacherName) {
  return schedules.filter(s => s.teacher === teacherName);
}

export function getUniqueSubjectsByTeacher(teacherName) {
  const subs = getSchedulesByTeacher(teacherName).map(s => s.subject);
  return [...new Set(subs)];
}

// ===== Subject Assignments helpers =====
export function getAllSubjectAssignments() {
  return subjectAssignments;
}

export function getAssignmentsByTeacher(teacherName) {
  return subjectAssignments.filter(a => a.teacher === teacherName);
}

export function getAssignmentsBySubject(subject) {
  return subjectAssignments.filter(a => a.subject === subject);
}

export function getAssignedStudentsFor(teacherName, subject) {
  const setIds = new Set(subjectAssignments
    .filter(a => a.teacher === teacherName && a.subject === subject)
    .map(a => a.studentId));
  return students.filter(s => setIds.has(s.id));
}

export function addAssignment(data) {
  const newAssign = { id: subjectAssignments.length + 1, ...data };
  subjectAssignments.push(newAssign);
  saveState();
  return newAssign;
}

export function deleteAssignment(id) {
  const idx = subjectAssignments.findIndex(a => a.id === id);
  if (idx !== -1) {
    const removed = subjectAssignments.splice(idx, 1)[0];
    saveState();
    return removed;
  }
  return null;
}

export function getGradeReport(className) {
  const classGrades = getGradesByClass(className);
  const subjects = [...new Set(classGrades.map(g => g.subject))];
  
  return subjects.map(subject => {
    const subjectGrades = classGrades.filter(g => g.subject === subject);
    const totalStudents = subjectGrades.length;
    const average = subjectGrades.reduce((sum, g) => sum + parseFloat(g.average), 0) / totalStudents;
    
    return {
      subject,
      totalStudents,
      average: average.toFixed(2),
      highest: Math.max(...subjectGrades.map(g => g.average)),
      lowest: Math.min(...subjectGrades.map(g => g.average))
    };
  });
}

// ===== FUNGSI UNTUK MENGOSONGKAN DATA SISWA =====
export function clearAllStudentData() {
  console.log('🚀 Memulai proses pengosongan data siswa...');
  
  // Simpan jumlah data sebelum dihapus untuk laporan
  const originalStudentsCount = students.length;
  const originalUsersCount = users.length;
  const originalGradesCount = grades.length;
  const originalAttendanceCount = attendance.length;
  const originalAssignmentsCount = subjectAssignments.length;
  const originalPaymentsCount = studentPayments.length;
  
  // Hapus semua data siswa dari array students
  students.length = 0;
  
  // Hapus semua user dengan role "siswa" dari array users
  users = users.filter(user => user.role !== 'siswa');
  const removedUsersCount = originalUsersCount - users.length;
  
  // Hapus semua data nilai siswa
  grades.length = 0;
  
  // Hapus semua data kehadiran siswa
  attendance.length = 0;
  
  // Hapus semua penugasan mata pelajaran siswa
  subjectAssignments.length = 0;
  
  // Hapus semua data tagihan siswa
  studentPayments.length = 0;
  
  // Simpan perubahan ke localStorage
  saveState();
  
  console.log('✅ Semua data siswa berhasil dihapus!');
  console.log(`📊 Data yang dihapus:`);
  console.log(`   - Data siswa: ${originalStudentsCount}`);
  console.log(`   - Akun siswa: ${removedUsersCount}`);
  console.log(`   - Data nilai: ${originalGradesCount}`);
  console.log(`   - Data kehadiran: ${originalAttendanceCount}`);
  console.log(`   - Penugasan mata pelajaran: ${originalAssignmentsCount}`);
  console.log(`   - Data tagihan: ${originalPaymentsCount}`);
  
  return {
    success: true,
    message: 'Semua data siswa berhasil dihapus',
    clearedData: {
      students: originalStudentsCount,
      studentUsers: removedUsersCount,
      grades: originalGradesCount,
      attendance: originalAttendanceCount,
      subjectAssignments: originalAssignmentsCount,
      payments: originalPaymentsCount
    }
  };
}

// Fungsi untuk mengosongkan data siswa dengan konfirmasi
export function clearAllStudentDataWithConfirmation() {
  const confirmed = confirm(
    '⚠️ PERINGATAN: Aksi ini akan menghapus SEMUA data siswa!\n\n' +
    'Data yang akan dihapus:\n' +
    '• Data siswa (biodata, NIS, kelas, dll)\n' +
    '• Akun login siswa\n' +
    '• Data nilai siswa\n' +
    '• Data kehadiran siswa\n' +
    '• Penugasan mata pelajaran siswa\n\n' +
    'Apakah Anda yakin ingin melanjutkan?'
  );
  
  if (confirmed) {
    return clearAllStudentData();
  } else {
    console.log('❌ Proses dibatalkan. Data siswa tidak dihapus.');
    return {
      success: false,
      message: 'Proses dibatalkan',
      clearedData: null
    };
  }
}

// ===== FUNGSI HELPER UNTUK LOGIN =====
export function getLoginCredentials() {
  return {
    admin: {
      email: "admin@smktanjung.ac.id",
      password: "admin123"
    },
    guru: [
      {
        email: "budi@smktanjung.ac.id",
        password: "guru123",
        name: "Budi Santoso"
      },
      {
        email: "sari@smktanjung.ac.id", 
        password: "guru123",
        name: "Sari Indah"
      }
    ],
    siswa: {
      email: "ahmad@smktanjung.ac.id",
      password: "siswa123"
    }
  };
}
